/**
 * 	Copyright 2011
 *  This file is part of JGooMap.
 *
 *  JGooMap is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  JGooMap is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with JGooMap.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Created on Aug 14, 2008
 * @author James Hansen(Kermode Bear Software)
 *
 */



/******************** Marker Management ******************************
 * The following html elements must be predefined:
 *
 * select_markers - Id of the select element.
 * marker_lat - Id of the Latitude text input element.
 * marker_lng  - Id of the Longitude text input element.
 * marker_info - Id of the marker info textarea element.
 * panTo - Id of the panto checkbox. Enables paning to the currently selected
 * 			marker from a selection list.
 *
 *********************************************************************/

/**
 *
 * @param {Object} map GMap object.
 */
function iGMarkerManager(map){
 	this.counter = 1; // Used for creating markers names on the fly.
 	this.markers = [];
	this.map = map;
	var that = this;

	google.maps.event.addListener(map, 'click', function(overlay, latlng, overlaylatlng){
		if(overlay instanceof google.maps.Marker){
			var i = findMarkerSelectIndex(overlay.myID);

			if (i != -1) {
				document.getElementById('select_markers').selectedIndex = i;
				that.markerSelect();
			}
		}
	});
}

/**
 * Adds a marker from the form fields.
 *
 * select_markers - Id of the select element.
 * marker_lat - Id of the Latitude text input element.
 * marker_lng  - Id of the Longitude text input element.
 *
 */
iGMarkerManager.prototype.addFormMarker = function(){
	var lat = document.getElementById('marker_lat').value;
	var lng = document.getElementById('marker_lng').value;

	if(!lat || !lng){
	 	alert("Please fill in the longitude and/or latitude of the marker!");
		return;
	}

	var latlng = new google.maps.LatLng(lat, lng);
	var info = document.getElementById('marker_info').value;

	this.addMarker(latlng, info);
}

/**
 * Adds a marker to the marker event manager.
 * @param {Object} latlng
 * @param {Object} info
 */
iGMarkerManager.prototype.addMarker = function(latlng, info){
	var name = 'Marker' + this.counter ;
	var options = { map: this.map, position: latlng, bouncy: true, draggable: true, title: name};
	var marker = new google.maps.Marker(options);

	marker.myID = this.counter;
	var infowindow = infoOps = null;
	if(info){
		var width = document.getElementById('marker_info_width').value;
		infoOps = { content : info, maxWidth : (width||200)  };
		infowindow = new google.maps.InfoWindow(infoOps);
		var that = this;
		var myListener = google.maps.event.addListener(marker, function(){
			infowindow.open(that.map, marker);
		});
		infowindow.open(that.map,marker);
		marker.myInfoListener = myListener;
	}

	this.markers[this.counter] = {
		marker: marker,
		info: null,
		infoWin: infowindow, 
		myOps : infoOps,
		title: name
	};

   	var option = document.createElement('option');
	var select = document.getElementById('select_markers');

	option.setAttribute('value', this.counter);
	option.innerHTML = name ;
	option.selected = true;
	select.appendChild(option);
	
	this.initEvents(marker);
	this.markerSelect();
	this.counter++;
}

/**
 * Initalize marker drag event.
 * @param {Object} marker
 */
iGMarkerManager.prototype.initEvents = function(marker){
	// update the marker from location
	google.maps.event.addListener(marker, 'drag', markerDrag);
	var that = this;
	google.maps.event.addListener(marker, 'click', function(){
		var i = findMarkerSelectIndex(this.myID);

		if (i != -1) {
			document.getElementById('select_markers').selectedIndex = i;
			that.markerSelect();
		}

	});
}

/**
 * Update the current selected marker.
 * marker_info - Id of the marker info textarea element.
 * select_markers - Id of the select element.
 */
iGMarkerManager.prototype.updateMarker = function(){
	var infoText = document.getElementById('marker_info').value;
	var selectEl = document.getElementById('select_markers');
	var selectedIndex = selectEl.selectedIndex;
	var that = this;
	
	if(selectedIndex == -1) return;
	var i = selectEl.options[selectedIndex].value;
	var title = $('marker_title').value;

	selectEl.options[ selectedIndex].innerHTML = title;


	if (this.markers[i].marker.getTitle() != title) {
		this.markers[i].title = title;
		this.markers[i].marker.setTitle(title);
	}

	if(infoText == ''){
		if(this.markers[i].infoWin){
			this.markers[i].infoWin.close();
		}
		this.markers[i].info = null;
		
	}else{
		var infoWin = null;
		var value = document.getElementById('marker_info_width').value;
		var ops = { maxWidth : ((value) ? value : 200) , content: infoText };
		this.markers[i].myOps = ops;
		if(this.markers[i].infoWin){
			this.markers[i].infoWin.setOptions(ops);
			infoWin = this.markers[i].infoWin;
		}else{
			
	
			if(this.markers[i].marker.myInfoListener)
				google.maps.event.removeListener(this.markers[i].marker.myInfoListener);
		
			infoWin = new google.maps.InfoWindow(ops);
			this.markers[i].infoWin = infoWin;
			this.markers[i].marker.myInfoListener = google.maps.event.addListener(this.markers[i].marker, 'click', function() {
				infoWin.open(that.map,that.markers[i].marker);
			});
		}
		
		infoWin.open(that.map, that.markers[i].marker);
	}
}

/**
 * Selects the current marker and update the form fields for that marker.
 * select_markers - Id of the select element.
 * marker_lat - Id of the Latitude text input element.
 * marker_lng  - Id of the Longitude text input element.
 */
iGMarkerManager.prototype.markerSelect = function (){
	var selectEl = document.getElementById('select_markers');
	var option = selectEl.options[selectEl.selectedIndex];
	var value = option.value;
	var marker = this.markers[value].marker;
	var latlng = marker.getPosition();
	$('marker_lat').value = latlng.lat();
	$('marker_lng').value = latlng.lng();
	
	$('marker_info').value = this.markers[value].infoWin ? (this.markers[value].infoWin.getContent() || '') : '';
	$('marker_title').value = this.markers[value].title;

	if(this.markers[value].infoWin && this.markers[value].infoWin.getContent())
		$('marker_info_width').value = this.markers[value].myOps.maxWidth;
	else
		$('marker_info_width').value = 200;

	this.panToMarker(option.value);
}

/**
 * Removes the selected marker in the from the map and form.
 * <select> Id of the select: 'select_markers'
 */
iGMarkerManager.prototype.removeMarker = function(){
	var selectEl = document.getElementById('select_markers');
	if(selectEl.selectedIndex == -1) return;
	var i = selectEl.options[selectEl.selectedIndex].value;
	this.markers[i].marker.setMap(null);
	delete this.markers[i].marker;
	delete this.markers[i];
	selectEl.remove(selectEl.selectedIndex);
}

/**
 * Pan to the marker selected by the index of the marker.
 * @param {Object} i
 */
iGMarkerManager.prototype.panToMarker = function(i){
	var value = document.getElementById('panToMarker').checked;

	if(value){
		this.map.panTo(this.markers[i].marker.getPosition());
	}
}

/**
 * Update marker form location fields.
 * @param {Object} latlng
 */
function markerDrag(event){
	var latlng = event.latLng;
	document.getElementById('marker_lat').value = latlng.lat();
	document.getElementById('marker_lng').value = latlng.lng();
}

/**
 *
 * @param {Object} id
 */
function findMarkerSelectIndex(id){
	var options = document.getElementById('select_markers').options;
	for(i =0; i < options.length; i++){
		if(options[i].value == id) return i;
	}

	return -1;
}