/**
 * 	Copyright 2011
 *  This file is part of JGooMap.
 *
 *  JGooMap is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  JGooMap is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with JGooMap.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Created on Aug 14, 2008
 * @author James Hansen(Kermode Bear Software)
 *
 */

/**
 * Builds javascript used for creating google maps. Mostly used to be insert into a web page via a WYSIWYG editor.
 *
 */
var iGMapBuilder = {};

/**
 * Creates javascript for a Google Map display from a iGMap object.
 * @param {Object} iGMap - iGMap Object.
 */
iGMapBuilder.buildMap = function(iGMap){
	var js = '';

	js += "var gmap = document.getElementById('gmap');\n";
	// GMap to GMap2 Version 2.x was having issue with the old GMap class. No Map Images,but Google Copyright 2011icons.
	js += 'var map = new google.maps.Map( gmap, ' + iGMapBuilder.buildOptions() + ' );\n';
	js += iGMapBuilder.buildBaseSettings();
	js += iGMapBuilder.buildMarkers(iGMap.iMarkerManager);
	js += iGMapBuilder.buildRoute(iGMap);
	return js + '\n';
};

/**
 *
 * @param {Object} iGMap
 */
iGMapBuilder.buildGoolgeMapInit = function(iGMap){
	var js = 'function initializeGoogleMap(){\n';
	js += iGMapBuilder.buildMap(iGMap);
	js += '}\n';
	js += 'google.maps.event.addDomListener(window, \'load\', initializeGoogleMap);\n';
	return js;
};

/**
 * Creates a control position.
 * @param {Object} id Id of the control position dropdown.
 */
iGMapBuilder.buildControlPosition = function(id){
	var el = document.getElementById(id);
	var position =  el.options[el.selectedIndex].value ;
	return 'google.maps.ControlPosition.' + position;
};

/**
 * Returns javascript for enabling a dragable map.
 * @param {Object} map - Google GMap2 object.
 */
iGMapBuilder.getDraggable = function(){
	var enable = document.getElementById('fixed').checked;
	return enable ? 'false' : 'true';
};

/**
 * Returns javascript for enable/disable zoom scrolling.
 */
iGMapBuilder.buildZoomScroll = function(){
	var enable = document.getElementById('zoomscrollable').checked;
	return enable ? 'true' : 'false';
};

/**
 * Returns javascript for the map info.
 * @param {Object} map
 */
iGMapBuilder.buildInfo = function(map){
	var mapinfo = document.getElementById('mapinfo').value;

	mapinfo = filterText(mapinfo);

	if(mapinfo != ''){
		var value = document.getElementById('info_width').value;
		var width = value ? value : 200;
		var ops = '{ maxWidth : ' + width + ', content: "' + mapinfo + '", ' +
					'position : map.getCenter()  }';
		
		var js = 'var infowindow = new google.maps.InfoWindow( ' + ops + ' );\n';

		js += 'infowindow.open( map );\n';
		
		return js;
	}

	return '';
};

/**
 * Calls other build javascript functions
 * @param {Object} map
 */
iGMapBuilder.buildBaseSettings = function(map){
	var js = '';
	js += iGMapBuilder.buildInfo(map);
	return js;
};

/**
 * Builds javascript options for a google map.
 */
iGMapBuilder.buildOptions = function(){
	var options = [];
	var width = document.getElementById('width').value || iGMap.DEFAULT_WIDTH;
	var height = document.getElementById('height').value || iGMap.DEFAULT_HEIGHT;
	var selectEl = document.getElementById('type');
	var type =  selectEl.options[selectEl.selectedIndex].value ;

	// Set up types.
	options[0] = 'mapTypeId: google.maps.MapTypeId.' + type + "\n" ;
	options[1] = 'center: ' + iGMapBuilder.getMapPosition() + "\n";
	options[2] = 'zoom: ' + iGMapBuilder.getZoomLevel() + "\n";
	options[3] = 'draggable: ' + iGMapBuilder.getDraggable() + "\n";
	options[4] = 'scrollwheel: ' + iGMapBuilder.buildZoomScroll() + "\n";
	options[5] = iGMapBuilder.buildControls() + "\n";
	
	return '{ ' + options.join(',') +  ' }' ;
};

/**
 * Builds javascript for a google marker icon.
 * @param {Object} marker - google.maps.Marker object.
 * @param {Object} info - Text of the marker info.
 */
iGMapBuilder.buildMarker = function(marker, info, title, ops,i){

	var js = '';
	js = 'var markerOps_' + i + ' = { map: map, position : ' + iGMapBuilder.getMarkerPosition(marker);
	js += ', title: "' + title + '" };\n';
	var markerVar = 'marker_' +  i ; 
	js += 'var ' + markerVar + ' = new google.maps.Marker( markerOps_' + i + ' );\n'; 

	if (info) {
		info = filterText( info );
		var opsStr = '{ maxWidth : ' +   ops.maxWidth  + ', content : "' + info + '" }';
		var infoVar = 'infowindow_' + i;
		js += 'var ' + infoVar +  ' = new google.maps.InfoWindow( ' + opsStr + ' );\n ';
		js += 'google.maps.event.addListener(' + markerVar + ', \'click\', function() {\n';
		js += infoVar + '.open(map,' + markerVar + ');\n';
		js +=  '});\n';
	}
	
	return js;
};

iGMapBuilder.buildMarkers = function (mkrMng){
	var controls = mkrMng.markers;
	var js = '';

	for(i = 0; i < controls.length; i++){
		if(controls[i]){
			var title = controls[i].title;
			var content = '' ;
			if(controls[i].infoWin){
				content = controls[i].infoWin.getContent();
			}
			
			js += iGMapBuilder.buildMarker(controls[i].marker, content , title, controls[i].myOps,i);
		}
	}

	return js;
};

/**
 * Builds the javascript for adding controls to a google map.
 */
iGMapBuilder.buildControls = function (){
	var fixed = document.getElementById('fixed').checked;
	var options = [];
	var scaleControl = iGMapBuilder.isRadio('scale');
	var mapTypeControl = iGMapBuilder.isRadio('type');
	var defaultControls = iGMapBuilder.isRadio('default');
	var smallControl = iGMapBuilder.isRadio('smallMap');
	var largeControl = iGMapBuilder.isRadio('largeMap');

	if(!fixed) {

		
		if(!defaultControls){
			options.push( 'disableDefaultUI: true\n');
			
		}else if(!smallControl){
			options.push(' navigationControlOptions: { position:' + iGMapBuilder.buildControlPosition('nav_position') + '}\n');
		}else if(!largeControl){
			options.push(' navigationControlOptions: { position:' + iGMapBuilder.buildControlPosition('largeMapControl_position') + '}\n');
		}
		
		if (smallControl) {
			var navOps = 'navigationControlOptions: { position:' + iGMapBuilder.buildControlPosition('smallMapControl_position');
			navOps += ',  style: google.maps.NavigationControlStyle.SMALL }\n';
			options.push(navOps );
			options.push('navigationControl: true\n');
		}else if(largeControl){
			options.push('navigationControl: true\n');
			options.push(' navigationControlOptions: { position:' + iGMapBuilder.buildControlPosition('largeMapControl_position') +
					', style : google.maps.NavigationControlStyle.DEFAULT' + '}\n');
		}
		

	}else{
		options.push( '\ndisableDefaultUI: true\n');
	}
	
	// Add map type control
	if (mapTypeControl) {
		options.push( 'mapTypeControl : true\n');
		options.push( 'mapTypeControlOptions: {  position:' + iGMapBuilder.buildControlPosition('type_position') + '}\n');
	}

	// Scale control
	if(scaleControl){
		options.push('scaleControl: true\n');
		options.push('scaleControlOptions : { position: ' + iGMapBuilder.buildControlPosition('scale_position') + '}\n');	
	}

	var js = options.join(', ');
	return js;
};

/**
 * Gets the map default position and return google javascript for google.maps.LatLng javascript.
 */
iGMapBuilder.getMapPosition = function(){
	var lat = document.getElementById('latitude').value || iGMap.DEFAULT_LATITUDE;
	var lng = document.getElementById('longitude').value || iGMap.DEFAULT_LONGITUDE;

	return 'new google.maps.LatLng( ' + lat + ', ' + lng + ')';
};

/**
 * Builds the javascript for a marker google.maps.LatLng position.
 * @param {Object} marker
 */
iGMapBuilder.getMarkerPosition = function(marker){
	var latlng = marker.getPosition();

	return 'new google.maps.LatLng( ' + latlng.lat() + ', ' + latlng.lng() + ')';
};


/**
 * Returns the zoom level for a map.
 * @param {Object} map
 */
iGMapBuilder.getZoomLevel = function(map){
	var level = document.getElementById('zoomlevel').value || iGMap.DEFAULT_ZOOMLEVEL;
	return level;
};


/**
 * Returns is an yes/no radio button group is selected.
 * @param {Object} id Prefix Id of the raido button group.
 */
iGMapBuilder.isRadio = function (id){
	var control = eval('document.mapform.' + id + '_control');

	for(i = 0; i < control.length; i++){
		if(control[i].checked == true ){
			if( control[i].value == '0' ) return false;
			else return true;
		}
	}
	return false;
};

/**
 * Displays a preview map in a separate window.
 * @param {Object} iMap - iGMap object.
 * @param {Object} key - Google map key.
 */
iGMapBuilder.previewGMap = function(iMap){

	var width = parseInt((document.getElementById('width').value || iGMap.DEFAULT_WIDTH));
	var windowWidth = width + 50;
	var height = parseInt((document.getElementById('height').value || iGMap.DEFAULT_HEIGHT));
	var windowHeight = height + 100 + ((iMap.isRoute()) ? 100 : 0);
	var html = '';
	html += ('<html><head><title>Preview Google Map</title>');
	html += ('<script type="text/javascript" src="http://maps.google.com/maps/api/js?sensor=true"><\/script>\n');
	html += '<meta http-equiv="refresh" content="10">\n';
	html += ('</head>' + "\n");

	html += ('<body>');

	html += ('<h1>Google Map Preview</h1>');
	html += ('<div style="margin: auto; width: ' + width + 'px;' + 'height:' + height + 'px;" id="gmap"></div>');

	if(iMap.isRoute()){
		html += ('<div id="directions"></div>');
	}
	html += ('<script type="text/javascript">');
//	html += ('function initGoogleMap() { \n');
	html += iGMapBuilder.buildMap(iMap);
//	html += (' }');
	html += ('</script>');
	html += ('</body></html>');
	
	var previewWindow = window.open('', 'previewWindow', 'scrollbars=yes,width=' + windowWidth + ',height=' + windowHeight );
	 previewWindow.document.write(html);
	// Ie Hack 
	if(window.ie) previewWindow.location.reload();
};

/**
 * Shows the script generated for iGMap object in a separate window.
 * @param {Object} iMap iGMap object.
 */
iGMapBuilder.previewScript = function(iMap){
	var previewWindow = window.open('', 'previewWindow', 'width=1024,height=700');
	var doc = previewWindow.document;
	doc.write('<pre>');
	var js = iGMapBuilder.buildMap(iMap)
	doc.write(js.replace(/\n/, '<br />'));
	doc.write('</pre>');
	previewWindow.focus();
};

iGMapBuilder.buildRoute = function(iMap){
	var js = '';

	if (iMap.isRoute()) {
		var start = document.getElementById('startpoint').value;
		var end = document.getElementById('endpoint').value;
		
		
		var options	= '{ ';
		options += iGMap.getDirectionRequest();
		options += ' }';
		js += 'var div_panel = document.getElementById(\'directions\');\n';
		js += 'var directionsService = new google.maps.DirectionsService(  );\n';
		js += 'var directionsDisplay = new google.maps.DirectionsRenderer(' + iMap.directionDisplayOptions.toString() + ');\n';
		js += 'directionsDisplay.setMap(map);\n';
		js += 'var directionRequest = ' + options + ';\n';
		js += 'directionsService.route(directionRequest, function(result, status) { \n';
		js += 'if (status == google.maps.DirectionsStatus.OK) { \n';
		js += '\t directionsDisplay.setDirections(result);\n } \n   });\n';
	}

	return js;
};


function filterText( text ){
	var newText = text.replace(/'/g, "\\'");
	return newText.replace( /\n/g, '<br>');
};