<?php
/**
* @package		EasyBlog
* @copyright	Copyright (C) 2010 Stack Ideas Private Limited. All rights reserved.
* @license		GNU/GPL, see LICENSE.php
* EasyBlog is free software. This version may have been modified pursuant
* to the GNU General Public License, and as distributed it includes or
* is derivative of works licensed under the GNU General Public License or
* other free or open source software licenses.
* See COPYRIGHT.php for copyright notices and details.
*/
defined('_JEXEC') or die('Restricted access');

require_once( JPATH_ROOT.DS.'components'.DS.'com_easyblog'.DS.'constants.php' );
require_once( EBLOG_HELPERS . DS . 'router.php' );

class TableECategory extends JTable
{
	/*
	 * The id of the category
	 * @var int
	 */
	var $id 						= null;

	/*
	 * The author of the category
	 * @var int
	 */
	var $created_by		= null;

	/*
	 * Category title
	 * @var string
	 */
	var $title					= null;

	/*
	 * Category title alias
	 * @var string
	 */
	var $alias					= null;

	/*
	 * Category avatar image filename
	 * @var string
	 */
	var $avatar					= null;

	/*
	 * Category parent_id
	 * @var int
	 */
	var $parent_id				= null;

	/*
	 * Category private
	 * @var int
	 */
	var $private				= null;

	/*
	 * Created datetime of the category
	 * @var datetime
	 */
	var $created				= null;

	/*
	 * Category status
	 * @var int
	 */
	var $status			= null;

	/*
	 * Category publishing status
	 * @var int
	 */
	var $published		= null;

	/*
	 * Category ordering
	 * @var int
	 */
	var $ordering		= null;

	/**
	 * @access	public
	 */
	var $description	= null;

	var $level			= null;
	var $lft			= null;
	var $rgt			= null;
	var $default		= null;

	/**
	 * Constructor for this class.
	 *
	 * @return
	 * @param object $db
	 */
	function __construct(& $db )
	{
		parent::__construct( '#__easyblog_category' , 'id' , $db );
	}

	function load( $key = null, $permalink = false )
	{
		if( !$permalink )
		{
			return parent::load( $key );
		}

		$db		= $this->getDBO();

		$query	= 'SELECT id FROM ' . $this->_tbl . ' '
				. 'WHERE alias=' . $db->Quote( $key );
		$db->setQuery( $query );

		$id		= $db->loadResult();

		// Try replacing ':' to '-' since Joomla replaces it
		if( !$id )
		{
			$query	= 'SELECT id FROM ' . $this->_tbl . ' '
					. 'WHERE alias=' . $db->Quote( JString::str_ireplace( ':' , '-' , $key ) );
			$db->setQuery( $query );

			$id		= $db->loadResult();
		}
		return parent::load( $id );
	}

	/**
	 * Overrides parent's delete method to add our own logic.
	 *
	 * @return boolean
	 * @param object $db
	 */
	function delete($pk = null)
	{
		$db		= $this->getDBO();
		$config = EasyBlogHelper::getConfig();

		$query	= 'SELECT COUNT(1) FROM ' . $db->nameQuote( '#__easyblog_post' ) . ' '
				. 'WHERE ' . $db->nameQuote( 'category_id' ) . '=' . $db->Quote( $this->id );
		$db->setQuery( $query );

		$count	= $db->loadResult();

		if( $count > 0 )
		{
			return false;
		}

		/* TODO */
		//remove avatar if previously already uploaded.
		$avatar = $this->avatar;

		if( $avatar != 'cdefault.png' && !empty($avatar))
		{

			$avatar_config_path = $config->get('main_categoryavatarpath');
			$avatar_config_path = rtrim($avatar_config_path, '/');
			$avatar_config_path = JString::str_ireplace('/', DS, $avatar_config_path);

			$upload_path		= JPATH_ROOT.DS.$avatar_config_path;

			$target_file_path		= $upload_path;
			$target_file 			= JPath::clean($target_file_path . DS. $avatar);

			if(JFile::exists( $target_file ))
			{
				JFile::delete( $target_file );
			}
		}

		return parent::delete();
	}

	function aliasExists()
	{
		$db		= $this->getDBO();

		$query	= 'SELECT COUNT(1) FROM ' . $db->nameQuote( '#__easyblog_category' ) . ' '
				. 'WHERE ' . $db->nameQuote( 'alias' ) . '=' . $db->Quote( $this->alias );

		if( $this->id != 0 )
		{
			$query	.= ' AND ' . $db->nameQuote( 'id' ) . '!=' . $db->Quote( $this->id );
		}
		$db->setQuery( $query );

		return $db->loadResult() > 0 ? true : false;
	}

	/**
	 * Overrides parent's bind method to add our own logic.
	 *
	 * @param Array $data
	 **/
	function bind( $data, $ignore = array() )
	{
		parent::bind( $data, $ignore );

		if( empty( $this->created ) )
		{
			$date			= JFactory::getDate();
			$this->created	= $date->toMySQL();
		}

		jimport( 'joomla.filesystem.filter.filteroutput');

		$i	= 1;
		while( $this->aliasExists() || empty($this->alias) )
		{
			$this->alias	= empty($this->alias) ? $this->title : $this->alias . '-' . $i;
			$i++;
		}

		//$this->alias 	= JFilterOutput::stringURLSafe( $this->alias );
		$this->alias 	= EasyBlogRouter::generatePermalink( $this->alias );
	}

	function getRSS()
	{
		return EasyBlogHelper::getHelper( 'Feeds' )->getFeedURL( 'index.php?option=com_easyblog&view=categories&id=' . $this->id, false, 'category' );
	}

	function getAtom()
	{
		return EasyBlogHelper::getHelper( 'Feeds' )->getFeedURL( 'index.php?option=com_easyblog&view=categories&id=' . $this->id , true, 'category' );
	}

	function getAvatar()
	{
		$avatar_link    = '';

		if($this->avatar == 'cdefault.png' || $this->avatar == 'default_category.png' || $this->avatar == 'components/com_easyblog/assets/images/default_category.png' || $this->avatar == 'components/com_easyblog/assets/images/cdefault.png' || empty($this->avatar))
		{
			$avatar_link   = 'components/com_easyblog/assets/images/default_category.png';
		}
		else
		{
			$avatar_link   = EasyImageHelper::getAvatarRelativePath('category') . '/' . $this->avatar;
		}

		return rtrim(JURI::root(), '/') . '/' . $avatar_link;
	}

	function getPostCount()
	{
		$db = JFactory::getDBO();

		$query  = 'SELECT count(1) FROM `#__easyblog_post` WHERE `category_id` = ' . $db->Quote($this->id);
		$db->setQuery($query);

		return $db->loadResult();
	}

	function getChildCount()
	{
		$db = JFactory::getDBO();

		$query  = 'SELECT count(1) FROM `#__easyblog_category` WHERE `parent_id` = ' . $db->Quote($this->id);
		$db->setQuery($query);

		return $db->loadResult();
	}

	/*
	 * Retrieves a list of active bloggers that contributed in this category.
	 *
	 * @param	null
	 * @return	Array	An array of TableProfile objects.
	 */
	public function getActiveBloggers()
	{
		$db		= JFactory::getDBO();
		$query	= 'SELECT DISTINCT(`created_by`) FROM ' . $db->nameQuote( '#__easyblog_post' ) . ' '
				. 'WHERE ' . $db->nameQuote( 'category_id' ) . '=' . $db->Quote( $this->id );
		$db->setQuery( $query );

		$rows		= $db->loadObjectList();

		if( !$rows )
		{
			return false;
		}

		$bloggers	= array();
		foreach( $rows as $row )
		{
			$profile	= EasyBlogHelper::getTable( 'Profile' , 'Table' );
			$profile->load( $row->created_by );

			$bloggers[]	= $profile;
		}

		return $bloggers;
	}

	public function store($updateNulls = false)
	{
		if( !empty( $this->created ))
		{
			$offset     	= EasyBlogDateHelper::getOffSet();
			$newDate		= JFactory::getDate( $this->created, $offset );
			$this->created  = $newDate->toMySQL();
		}
		else
		{
			$newDate		= JFactory::getDate();
			$this->created  = $newDate->toMySQL();
		}

		// Figure out the proper nested set model
		if( $this->id == 0 && $this->lft == 0 )
		{
			// No parent id, we use the current lft,rgt
			if( $this->parent_id )
			{
				$left           = $this->getLeft( $this->parent_id );
				$this->lft      = $left;
				$this->rgt      = $this->lft + 1;

				// Update parent's right
				$this->updateRight( $left );
				$this->updateLeft( $left );
			}
			else
			{
				$this->lft      = $this->getLeft() + 1;
				$this->rgt      = $this->lft + 1;
			}
		}

		return parent::store($updateNulls);
	}

	public function saveACL( $post )
	{

		$catRuleItems	= EasyBlogHelper::getTable( 'ECategoryAclItem' , 'Table' );
		$categoryRules  = $catRuleItems->getAllRuleItems();

		foreach( $categoryRules as $rule)
		{
			$key    = 'category_acl_'.$rule->action;
			if( isset( $post[ $key ] ) )
			{
				if( count( $post[ $key ] ) > 0)
				{
					foreach( $post[ $key ] as $joomla)
					{
						//now we reinsert again.
						$catRule	= EasyBlogHelper::getTable( 'ECategoryAcl' , 'Table' );
						$catRule->category_id	= $this->id;
						$catRule->acl_id 		= $rule->id;
						$catRule->type 			= 'group';
						$catRule->content_id 	= $joomla;
						$catRule->status 		= '1';
						$catRule->store();
					} //end foreach

				} //end if
			}//end if
		}
	}

	public function deleteACL( $aclId = '' )
	{
		$db = JFactory::getDBO();

		$query  = 'delete from `#__easyblog_category_acl`';
		$query	.= ' where `category_id` = ' . $db->Quote( $this->id );
		if( !empty($aclId) )
			$query	.= ' and `acl_id` = ' . $db->Quote( $aclId );

		$db->setQuery( $query );
		$db->query();

		return true;
	}

	public function getAssignedACL()
	{
		$db = JFactory::getDBO();

		$acl    = array();

		$query  = 'SELECT a.`category_id`, a.`content_id`, a.`status`, b.`id` as `acl_id`';
		$query  .= ' FROM `#__easyblog_category_acl` as a';
		$query  .= ' LEFT JOIN `#__easyblog_category_acl_item` as b';
		$query  .= ' ON a.`acl_id` = b.`id`';
		$query  .= ' WHERE a.`category_id` = ' . $db->Quote( $this->id );
		$query  .= ' AND a.`type` = ' . $db->Quote( 'group' );

		//echo $query;

		$db->setQuery( $query );
		$result = $db->loadObjectList();


		$joomlaGroups    = EasyBlogHelper::getJoomlaUserGroups();

		if( EasyBlogHelper::getJoomlaVersion() < '1.6' )
		{
			$guest  = new stdClass();
			$guest->id		= '0';
			$guest->name	= 'Public';
			$guest->level	= '0';
			array_unshift($joomlaGroups, $guest);
		}

		$acl             = $this->_mapRules($result, $joomlaGroups);

		return $acl;

	}

	public function _mapRules( $catRules, $joomlaGroups)
	{
		$db 	= JFactory::getDBO();
		$acl    = array();

		$query  = 'select * from `#__easyblog_category_acl_item` order by id';
		$db->setQuery( $query );

		$result = $db->loadObjectList();

		if( !$result )
		{
			return $result;
		}

		foreach( $result as $item )
		{
			$aclId 		= $item->id;
			$default    = $item->default;

			foreach( $joomlaGroups as $joomla )
			{
				$groupId    	= $joomla->id;
				$catRulesCnt    = count($catRules);

				if ( empty($acl[$aclId][$groupId]) )
				{
					$acl[$aclId][$groupId] = new stdClass();
				}

				//now match each of the catRules
				if( $catRulesCnt > 0)
				{
					$cnt    = 0;
					foreach( $catRules as $rule)
					{
						if($rule->acl_id == $aclId && $rule->content_id == $groupId)
						{
							$acl[$aclId][$groupId]->status  	= $rule->status;
							$acl[$aclId][$groupId]->acl_id  	= $aclId;
							$acl[$aclId][$groupId]->groupname	= $joomla->name;
							$acl[$aclId][$groupId]->groupid		= $groupId;
							break;
						}
						else
						{
							$cnt++;
						}
					}

					if( $cnt == $catRulesCnt)
					{
						//this means the rules not exist in this joomla group.
						$acl[$aclId][$groupId]->status  	= '0';
						$acl[$aclId][$groupId]->acl_id  	= $aclId;
						$acl[$aclId][$groupId]->groupname	= $joomla->name;
						$acl[$aclId][$groupId]->groupid		= $groupId;
					}
				}
				else
				{
					$acl[$aclId][$groupId]->status  	= $default;
					$acl[$aclId][$groupId]->acl_id  	= $aclId;
					$acl[$aclId][$groupId]->groupname	= $joomla->name;
					$acl[$aclId][$groupId]->groupid		= $groupId;
				}
			}
		}

		return $acl;
	}

	public function checkPrivacy()
	{
		$obj			= new EasyBlogPrivacyError();
		$obj->allowed	= true;

		$my 			= JFactory::getUser();

		if( $this->private == '1' && $my->id == 0)
		{
			$obj->allowed	= false;
			$obj->error		= EasyBlogPrivacyHelper::getErrorHTML();
		}
		else
		{
			if( $this->private == '2')
			{
				$cats    = EasyBlogHelper::getPrivateCategories();

				if( in_array($this->id, $cats) )
				{
					$obj->allowed	= false;
					$obj->error		= JText::_( 'COM_EASYBLOG_PRIVACY_JOMSOCIAL_NOT_AUTHORIZED_ERROR' );
				}

			}
		}

		return $obj;
	}

	// category ordering with lft and rgt
	public function updateLeft( $left, $limit = 0 )
	{
		$db     = JFactory::getDBO();
		$query  = 'UPDATE ' . $db->nameQuote( $this->_tbl ) . ' '
				. 'SET ' . $db->nameQuote( 'lft' ) . '=' . $db->nameQuote( 'lft' ) . ' + 2 '
				. 'WHERE ' . $db->nameQuote( 'lft' ) . '>=' . $db->Quote( $left );

		if( !empty( $limit ) )
			$query  .= ' and `lft`  < ' . $db->Quote( $limit );

		$db->setQuery( $query );
		$db->Query();
	}

	public function updateRight( $right, $limit = 0 )
	{
		$db     = JFactory::getDBO();
		$query  = 'UPDATE ' . $db->nameQuote( $this->_tbl ) . ' '
				. 'SET ' . $db->nameQuote( 'rgt' ) . '=' . $db->nameQuote( 'rgt' ) . ' + 2 '
				. 'WHERE ' . $db->nameQuote( 'rgt' ) . '>=' . $db->Quote( $right );

		if( !empty( $limit ) )
			$query  .= ' and `rgt`  < ' . $db->Quote( $limit );

		$db->setQuery( $query );
		$db->Query();
	}

	public function getLeft( $parent = 0 )
	{
		$db     = JFactory::getDBO();

		if( $parent != 0 )
		{
			$query  = 'SELECT `rgt`' . ' '
					. 'FROM ' . $db->nameQuote( $this->_tbl ) . ' '
					. 'WHERE ' . $db->nameQuote( 'id' ) . '=' . $db->Quote( $parent );
		}
		else
		{
			$query  = 'SELECT MAX(' . $db->nameQuote( 'rgt' ) . ') '
					. 'FROM ' . $db->nameQuote( $this->_tbl );
		}
		$db->setQuery( $query );

		$left   = (int) $db->loadResult();

		return $left;
	}

	function move( $direction, $where = '' )
	{
		$db = JFactory::getDBO();

		if( $direction == -1) //moving up
		{
			// getting prev parent
			$query  = 'select `id`, `lft`, `rgt` from `#__easyblog_category` where `lft` < ' . $db->Quote($this->lft);
			if($this->parent_id == 0)
				$query  .= ' and parent_id = 0';
			else
				$query  .= ' and parent_id = ' . $db->Quote($this->parent_id);
			$query  .= ' order by lft desc limit 1';

			//echo $query;exit;
			$db->setQuery($query);
			$preParent  = $db->loadObject();

			// calculating new lft
			$newLft = $this->lft - $preParent->lft;
			$preLft = ( ($this->rgt - $newLft) + 1) - $preParent->lft;

			//get prevParent's id and all its child ids
			$query  = 'select `id` from `#__easyblog_category`';
			$query  .= ' where lft >= ' . $db->Quote($preParent->lft) . ' and rgt <= ' . $db->Quote($preParent->rgt);
			$db->setQuery($query);

			//echo '<br>' . $query;
			$preItemChilds = $db->loadResultArray();
			$preChildIds   = implode(',', $preItemChilds);
			$preChildCnt   = count($preItemChilds);

			//get current item's id and it child's id
			$query  = 'select `id` from `#__easyblog_category`';
			$query  .= ' where lft >= ' . $db->Quote($this->lft) . ' and rgt <= ' . $db->Quote($this->rgt);
			$db->setQuery($query);

			//echo '<br>' . $query;
			$itemChilds = $db->loadResultArray();
			$childIds   = implode(',', $itemChilds);
			$ChildCnt   = count($itemChilds);

			//now we got all the info we want. We can start process the
			//re-ordering of lft and rgt now.
			//update current parent block
			$query  = 'update `#__easyblog_category` set';
			$query  .= ' lft = lft - ' . $db->Quote($newLft);
			if( $ChildCnt == 1 ) //parent itself.
			{
				$query  .= ', `rgt` = `lft` + 1';
			}
			else
			{
				$query  .= ', `rgt` = `rgt` - ' . $db->Quote($newLft);
			}
			$query  .= ' where `id` in (' . $childIds . ')';

			//echo '<br>' . $query;
			$db->setQuery($query);
			$db->query();

			$query  = 'update `#__easyblog_category` set';
			$query  .= ' lft = lft + ' . $db->Quote($preLft);
			$query  .= ', rgt = rgt + ' . $db->Quote($preLft);
			$query  .= ' where `id` in (' . $preChildIds . ')';

			//echo '<br>' . $query;
			//exit;
			$db->setQuery($query);
			$db->query();

			//now update the ordering.
			$query  = 'update `#__easyblog_category` set';
			$query  .= ' `ordering` = `ordering` - 1';
			$query  .= ' where `id` = ' . $db->Quote($this->id);
			$db->setQuery($query);
			$db->query();

			//now update the previous parent's ordering.
			$query  = 'update `#__easyblog_category` set';
			$query  .= ' `ordering` = `ordering` + 1';
			$query  .= ' where `id` = ' . $db->Quote($preParent->id);
			$db->setQuery($query);
			$db->query();

			return true;
		}
		else //moving down
		{
			// getting next parent
			$query  = 'select `id`, `lft`, `rgt` from `#__easyblog_category` where `lft` > ' . $db->Quote($this->lft);
			if($this->parent_id == 0)
				$query  .= ' and parent_id = 0';
			else
				$query  .= ' and parent_id = ' . $db->Quote($this->parent_id);
			$query  .= ' order by lft asc limit 1';

			$db->setQuery($query);
			$nextParent  = $db->loadObject();


			$nextLft 	= $nextParent->lft - $this->lft;
			$newLft 	= ( ($nextParent->rgt - $nextLft) + 1) - $this->lft;


			//get nextParent's id and all its child ids
			$query  = 'select `id` from `#__easyblog_category`';
			$query  .= ' where lft >= ' . $db->Quote($nextParent->lft) . ' and rgt <= ' . $db->Quote($nextParent->rgt);
			$db->setQuery($query);

			//echo '<br>' . $query;
			$nextItemChilds = $db->loadResultArray();
			$nextChildIds   = implode(',', $nextItemChilds);
			$nextChildCnt   = count($nextItemChilds);

			//get current item's id and it child's id
			$query  = 'select `id` from `#__easyblog_category`';
			$query  .= ' where lft >= ' . $db->Quote($this->lft) . ' and rgt <= ' . $db->Quote($this->rgt);
			$db->setQuery($query);

			//echo '<br>' . $query;
			$itemChilds = $db->loadResultArray();
			$childIds   = implode(',', $itemChilds);

			//now we got all the info we want. We can start process the
			//re-ordering of lft and rgt now.

			//update next parent block
			$query  = 'update `#__easyblog_category` set';
			$query  .= ' `lft` = `lft` - ' . $db->Quote($nextLft);
			if( $nextChildCnt == 1 ) //parent itself.
			{
				$query  .= ', `rgt` = `lft` + 1';
			}
			else
			{
				$query  .= ', `rgt` = `rgt` - ' . $db->Quote($nextLft);
			}
			$query  .= ' where `id` in (' . $nextChildIds . ')';

			//echo '<br>' . $query;
			$db->setQuery($query);
			$db->query();

			//update current parent
			$query  = 'update `#__easyblog_category` set';
			$query  .= ' lft = lft + ' . $db->Quote($newLft);
			$query  .= ', rgt = rgt + ' . $db->Quote($newLft);
			$query  .= ' where `id` in (' . $childIds. ')';

			//echo '<br>' . $query;
			//exit;

			$db->setQuery($query);
			$db->query();

			//now update the ordering.
			$query  = 'update `#__easyblog_category` set';
			$query  .= ' `ordering` = `ordering` + 1';
			$query  .= ' where `id` = ' . $db->Quote($this->id);

			//echo '<br>' . $query;

			$db->setQuery($query);
			$db->query();

			//now update the previous parent's ordering.
			$query  = 'update `#__easyblog_category` set';
			$query  .= ' `ordering` = `ordering` - 1';
			$query  .= ' where `id` = ' . $db->Quote($nextParent->id);

			//echo '<br>' . $query;

			$db->setQuery($query);
			$db->query();

			return true;
		}
	}

	public function rebuildOrdering($parentId = null, $leftId = 0 )
	{
		$db = JFactory::getDBO();

		$query  = 'select `id` from `#__easyblog_category`';
		$query  .= ' where parent_id = ' . $db->Quote( $parentId );
		$query  .= ' order by lft';

		$db->setQuery( $query );
		$children = $db->loadObjectList();

		// The right value of this node is the left value + 1
		$rightId = $leftId + 1;

		// execute this function recursively over all children
		foreach ($children as $node)
		{
			// $rightId is the current right value, which is incremented on recursion return.
			// Increment the level for the children.
			// Add this item's alias to the path (but avoid a leading /)
			$rightId = $this->rebuildOrdering($node->id, $rightId );

			// If there is an update failure, return false to break out of the recursion.
			if ($rightId === false) return false;
		}

		// We've got the left value, and now that we've processed
		// the children of this node we also know the right value.
		$updateQuery    = 'update `#__easyblog_category` set';
		$updateQuery    .= ' `lft` = ' . $db->Quote( $leftId );
		$updateQuery    .= ', `rgt` = ' . $db->Quote( $rightId );
		$updateQuery    .= ' where `id` = ' . $db->Quote($parentId);

		$db->setQuery($updateQuery);

		// If there is an update failure, return false to break out of the recursion.
		if (! $db->query())
		{
			return false;
		}

		// Return the right value of this node + 1.
		return $rightId + 1;
	}

}
